﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using System.Xml.Serialization;
using Psion.RFID.HF;

namespace HF_Demo
{
    public sealed partial class ScriptForm : Form
    {
        private readonly Reader _reader; //RFID Reader instance
        private readonly List<Script> _detectedScripts;
        private Script _currentScript;

        public ScriptForm(
            Reader reader)
        {
            InitializeComponent();

            if (!Device.IsWM) mainMenu1.Dispose();

            Height = Screen.PrimaryScreen.WorkingArea.Height;
            _detectedScripts = new List<Script>();
            _currentScript = new Script();
            _reader = reader;

            comboBoxDisplayFormat.SelectedIndex = 0; // select HEX format display
        }


        /// <summary>
        /// Load focused script in current script
        /// </summary>
        public void LoadSelectedScript()
        {
            try
            {
                if (lstVw_Scripts.FocusedItem.Focused)
                {
                    _currentScript = _detectedScripts[lstVw_Scripts.FocusedItem.Index];
                }
                else
                {
                    PrintLine("< " + "No Script Selected", Color.Red);
                }
            }
            catch (Exception)
            {
                PrintLine("< " + "Error While Loading Script", Color.Red);
            }
        }

        /// <summary>
        /// Retrieves all the script files in located in /Script folder
        /// </summary>
        public void RetrieveScripts()
        {
            try
            {
                //Create the XML Serializer

                var xmlSerializer = new XmlSerializer(new Script().GetType());
                string strScriptPath = IOHelpers.ApplicationPath + @"/Scripts";
                try
                {
                    var di = new DirectoryInfo(strScriptPath);
                    FileInfo[] fileInfo = di.GetFiles("*.xml");

                    foreach (FileInfo fiTemp in fileInfo)
                    {
                        var fileStream = new FileStream(fiTemp.FullName, FileMode.Open);

                        _detectedScripts.Add((Script) xmlSerializer.Deserialize(fileStream));
                        fileStream.Close();
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        /// <summary>
        /// Load detected scipts in script listview
        /// </summary>
        public void LoadDetectedScripts()
        {
            int i = 0;
            foreach (Script scrpt in _detectedScripts)
            {
                lstVw_Scripts.Items.Add(new ListViewItem(scrpt.Name));
                lstVw_Scripts.Items[i].SubItems.Add(scrpt.ScriptDescription);
                i++;
            }
        }

        /// <summary>
        /// Checks if results returned by reader is as expected in script
        /// </summary>
        /// <param name="t">single task</param>
        /// <param name="readerReturnValue">actual return value</param>
        private void resultChecker(Script.SingleTask t, string readerReturnValue)
        {
            string expectedValue = t.ReturnValue.ToLower();
            string actualValue = readerReturnValue.ToLower();

            if (t.ReturnValueCheck)
                if (t.ReturnValue.Where((t1, i) => expectedValue[i] != 'x' && expectedValue[i] != actualValue[i]).Any())
                {
                    throw new EvaluateException(string.Format("Return Value [{0}] != from script [{1}]", 
                        readerReturnValue,
                        t.ReturnValue));
                }

            if (t.LengthCheck)
            {
                if (expectedValue.Length != actualValue.Length)
                    throw new EvaluateException(string.Format("Response Lgth [{0}] != from script [{1}]",
                        actualValue.Length,
                        expectedValue.Length));
            }
        }


        /// <summary>
        /// Executes all actions in a task
        /// </summary>
        /// <param name="t"></param>
        /// <returns></returns>
        private string RunTask(Script.SingleTask t)
        {
            //PrintLine("** " + t.Name + " **");
            //PrintLine(t.TaskDescription);
            PrintLine("TX > " + t.Command + " " + t.Data, Color.FromArgb(0, 140, 24)); //Readible Green to higlight commands

            string bufferHexString = string.Empty;

            try
            {
                // Check no empty command
                if (t.Command.Equals("")) throw new Exception("Empty Cmd not allowed");

                // convert data from HEX string to byte array
                byte[] data = ConvertHelpers.ConvertHexStringToByteArray(t.Data);

                var sw = new Stopwatch();

                sw.Start();
                byte[] buffer = _reader.Protocol(t.Command, data);
                sw.Stop();

                // check possible error
                RFIDHelpers.CheckErrors(buffer);

                // convert result in Hex-string
                bufferHexString = ConvertHelpers.ConvertByteArrayToHexString(buffer);
                string bufferString = ConvertHelpers.HexConvertToString(buffer, "");

                // checks returned value with the expected from Script
                // this can check difference of length / difference of 
                resultChecker(t,
                              (t.ReturnValueHexFormat ? bufferHexString : bufferString));

                // show execution timing
                if (comboBoxDisplayFormat.SelectedIndex == 0 || // HEX format display 
                    comboBoxDisplayFormat.SelectedIndex == 2)
                    PrintLine(string.Format("RX[{1:000}] HEX < {0} ", bufferHexString, sw.ElapsedMilliseconds), Color.RoyalBlue);


                if (comboBoxDisplayFormat.SelectedIndex == 1 || // ASCII format display 
                    comboBoxDisplayFormat.SelectedIndex == 2)
                    PrintLine(string.Format("RX[{1:000}]ASCII< {0} ", bufferString, sw.ElapsedMilliseconds), Color.RoyalBlue);

            }
            catch (InvalidCastException invalidCastException)
            {
                PrintLine("< " + invalidCastException.Message, Color.Red);
                PrintLine("Should be hexadecimal value [0..9A..F]", Color.OrangeRed);

                if (checkBoxStopAfterException.Checked) throw;
            }
            catch (Exception ex)
            {
                PrintLine("< " + ex.Message, Color.Red);

                if (checkBoxStopAfterException.Checked) throw;
            }

            // we will save the response of the command
            return bufferHexString;
        }

        #region Events

        /// <summary>
        /// Run a command/script
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtRunClick(object sender, EventArgs e)
        {
            // clear listview & print list
            Clear();

            //Set stopwatch
            var sw = new Stopwatch();

            LoadSelectedScript();
            if (_currentScript.Tasks.Count == 0)
            {
                PrintLine("< " + "WARN.: No Tasks Detected in Script", Color.Orange);
            }

            else
            {
                //Send command
                string returnValue = string.Empty;

                PrintLine("***** Starting '" + _currentScript.Name + "' script *****", Color.Yellow, Color.DarkGray);
                ShowPrint();

                sw.Start();
                //Run every task in script
                for (int i = 0; i < _currentScript.Tasks.Count; i++)
                {
                    PrintLine("** " + _currentScript.Tasks[i].Name + " **", Color.Black, Color.LightGray);
                    if (! _currentScript.Tasks[i].TaskDescription.Equals("") &&
                        ! _currentScript.Tasks[i].TaskDescription.Equals("No desc.")) 
                        PrintLine(_currentScript.Tasks[i].TaskDescription, Color.Gray);
                    
                    if (_currentScript.Tasks[i].Command == "" || _currentScript.Tasks[i].Command == " ")
                    {
                        PrintLine("< WARN: No Command detected.", Color.DarkOrange);
                        continue;
                    }

                    if (i > 0)
                    {
                        if (_currentScript.Tasks[i].UsePreviousValue) //Use previous command result
                        {
                            int stripLead = _currentScript.Tasks[i].StripLead*2;
                            int stripTrail = _currentScript.Tasks[i].StripTrail*2;
                            if (returnValue != "")
                            {
                                if (stripLead + stripTrail > returnValue.Length)
                                {
                                    PrintLine("StripLead/Trail Value Too High", Color.Red);
                                    return;
                                }
                                string tempVal = returnValue.Substring(stripLead,
                                                                       returnValue.Length - stripLead - stripTrail);
                                _currentScript.Tasks[i].Data += tempVal;
                            }
                            else
                            {
                                PrintLine("WARN: Prev. task return val. empty", Color.DarkOrange);
                            }
                        }
                    }

                    try
                    {
                        returnValue = RunTask(_currentScript.Tasks[i]);
                    }
                    catch (Exception)
                    {
                        break; // stop script execution
                    }
                }


                //Stop stopwatch and display the elapsed time
                sw.Stop();
                InsertPrintLine(1, string.Format("*** Script Executed in {0:000} ms", sw.ElapsedMilliseconds), Color.White, Color.DarkGray);
                PrintLine("******** End Of Script ********", Color.White, Color.DarkGray);
            }

            ShowPrint();
        }

        private class Line
        {
            public string Message;
            public Color ForeColor = Color.Black;
            public Color BackColor = Color.White;
        }
        readonly List<Line> _lines = new List<Line>();
        private void Clear()
        {
            _lines.Clear();
            lstVw_Cnsl.Items.Clear();
            lstVw_Cnsl.Refresh();
        }
        private void PrintLine(string message)
        { 
            PrintLine(message, Color.Black, Color.White);
        }
        private void PrintLine(string message, Color foreColor)
        {
            PrintLine(message, foreColor, Color.White);
        }
        private void PrintLine(string message, Color foreColor, Color backColor)
        {
            _lines.Add(new Line { Message = message, ForeColor = foreColor, BackColor = backColor });
        }
        private void InsertPrintLine(int index, string message, Color foreColor, Color backColor)
        {
            _lines.Insert(index, new Line { Message = message, ForeColor = foreColor, BackColor = backColor });
        }
        private void ShowPrint()
        {
            lstVw_Cnsl.Items.Clear();
            foreach (var line in _lines)
            {
                lstVw_Cnsl.Items.Add(new ListViewItem(line.Message));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = line.ForeColor;
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].BackColor = line.BackColor;

                lstVw_Cnsl.Refresh();
            }
        }
        
        /// <summary>
        /// Handles the Load event of the HyperTerminalForm control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void HyperTerminalFormLoad(object sender, EventArgs e)
        {
            RetrieveScripts();
            LoadDetectedScripts();
        }

        /// <summary>
        /// Handles the SelectedIndexChanged event of the lstVw_Scripts control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void LstVwScriptsSelectedIndexChanged(object sender, EventArgs e)
        {
            bt_Run.Enabled = true;
        }

        /// <summary>
        /// Handles the Click event of the menuItem1 control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void MenuItem1Click(object sender, EventArgs e)
        {
            Close();
        }
        #endregion
    }
}